<?php
/**
 * Copyright © 2018 Magenest. All rights reserved.
 * See COPYING.txt for license details.
 *
 * Magenest_QuickBooksDesktop extension
 * NOTICE OF LICENSE
 */

namespace Magenest\QuickBooksDesktop\Model\QBXML;

use Magenest\QuickBooksDesktop\Model\Config\Source\Version;
use Magento\Sales\Model\Order\Creditmemo as CreditmemoModel;
use Magenest\QuickBooksDesktop\Model\QBXML;
use Magenest\QuickBooksDesktop\Model\Mapping;
use Magenest\QuickBooksDesktop\Helper\CreateQueue as QueueHelper;
use \Magento\Framework\ObjectManagerInterface;

/**
 * Class CreditMemo
 *
 * @package Magenest\QuickBooksDesktop\Model\QBXML
 */
class CreditMemo extends QBXML
{
    /**
     * @var QueueHelper
     */
    protected $_queueHelper;

    /**
     * @var CreditmemoModel
     */
    protected $_creditMemo;

    /**
     * @var Mapping
     */
    public $_map;

    /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    public $scopeConfig;

    /**
     * @var \Magento\Catalog\Model\ProductFactory
     */
    public $_productFactory;

    /**
     * @var \Magento\Customer\Model\CustomerFactory
     */
    protected $customerFactory;

    /**
     * CreditMemo constructor.
     * @param CreditmemoModel $creditmemo
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     * @param Mapping $map
     * @param \Magento\Catalog\Model\ProductFactory $productFactory
     * @param ObjectManagerInterface $objectManager
     * @param QueueHelper $queueHelper
     */
    public function __construct(
        CreditmemoModel $creditmemo,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        Mapping $map,
        \Magento\Catalog\Model\ProductFactory $productFactory,
        ObjectManagerInterface $objectManager,
        \Magento\Customer\Model\CustomerFactory $customerFactory,
        QueueHelper $queueHelper
    )
    {
        parent::__construct($objectManager);
        $this->_creditMemo = $creditmemo;
        $this->scopeConfig = $scopeConfig;
        $this->_map = $map;
        $this->_productFactory = $productFactory;
        $this->_queueHelper = $queueHelper;
        $this->_version = $this->_queueHelper->getQuickBooksVersion();
        $this->customerFactory = $customerFactory;
    }

    /**
     * Get XML using sync to QBD
     *
     * @param $id
     * @return string
     */
    public function getXml($id)
    {
        /** @var \Magento\Sales\Model\Order\Creditmemo $model */
        $model = $this->_creditMemo->load($id);
        /** @var \Magento\Sales\Model\Order $order */
        $order = $model->getOrder();
        $billAddress = $model->getBillingAddress();
        $shipAddress = $model->getShippingAddress();
        $customerName = $order->getCustomerName();
        $customerId = $model->getOrder()->getCustomerId();
        if ($customerId) {
            $customer = $this->customerFactory->create()->load($customerId);
            if ($customer->getEntityId()) {
                $customerReceive = $customer->getName() . ' ' . $customerId;
            } else {
                $customerReceive = $customerName . ' ' . $customerId;
            }
        } else {
            $customerReceive = $billAddress->getName() . ' ' . $order->getIncrementId();
        }
        $xml = $this->multipleXml($customerReceive, ['CustomerRef', 'FullName'], 40);
        $date = date("Y-m-d", strtotime($model->getCreatedAt()));
        $xml .= $this->simpleXml($date, 'TxnDate');
        $xml .= $this->simpleXml($model->getIncrementId(), 'RefNumber', 11);
        $xml .= $billAddress ? $this->getAddress($billAddress) : '';
        $xml .= $shipAddress ? $this->getAddress($shipAddress, 'ship') : '';
        $shipMethod = strtok($order->getShippingMethod(), '_');

        if (!empty($shipMethod)) {
            $xml .= $this->multipleXml($shipMethod, ['ShipMethodRef', 'FullName'], 15);
        }

        $taxCode = $this->getTaxCodeTransaction($order);

        if ($taxCode != null && $this->_version == Version::VERSION_US) {
            $xml .= $this->multipleXml($taxCode, ['ItemSalesTaxRef', 'FullName'], 30);
        }

        /** @var \Magento\Sales\Model\Order\Creditmemo\Item $item */
        foreach ($model->getAllItems() as $item) {
            $item_id = $order->getItemsCollection()->addFieldToFilter('sku', $item->getSku())->getLastItem()->getItemId();
            $xml .= $this->getOrderItem($item, $item_id);
        }

        if ($model->getShippingAmount() != 0) {
            $dataShipping =
                [
                    'type' => 'Shipping',
                    'desc' => $order->getShippingDescription(),
                    'rate' => $model->getShippingAmount(),
                    'tax_amount' => $model->getShippingTaxAmount(),
                    'txn_id' => null,
                    'taxcode' => $taxCode
                ];

            $xml .= $this->getOtherItem($dataShipping, 'CreditMemoLineAdd');
        }

        if ($model->getDiscountAmount() != 0) {
            $discount = $model->getDiscountAmount();
            if ($discount < 0) {
                $discount = 0 - $discount;
            }
            $dataDiscount =
                [
                    'type' => 'Discount',
                    'desc' => $model->getDiscountDescription(),
                    'rate' => $discount,
                    'tax_amount' => $model->getDiscountTaxCompensationAmount(),
                    'txn_id' => null,
                    'taxcode' => $taxCode
                ];
            $xml .= $this->getOtherItem($dataDiscount, 'CreditMemoLineAdd');
        }


        return $xml;
    }

    /**
     * Get Order Item XML
     *
     * @param \Magento\Sales\Model\Order\Creditmemo\Item $item *
     * @return string
     */
    protected function getOrderItem($item, $item_id)
    {
        $price = $item->getPrice();
        $taxAmount = $item->getTaxAmount();
        $qty = $item->getQty();

        if($item->getOrderItem()->getProductType() == 'bundle' || $item->getPrice() == 0){
            return;
        }

        if($item->getOrderItem()->getParentItem() && $item->getOrderItem()->getParentItem()->getProductType() == 'configurable'){
            return;
        }

        if($item->getOrderItem()->getProductType() == 'configurable'){
            if(isset($item->getOrderItem()->getChildrenItems()[0])){
                $item->setProductId($item->getOrderItem()->getChildrenItems()[0]->getProductId());
            }
        }

        $product = $this->_productFactory->create()->load($item->getProductId());
        $sku = $product->getSku();

        if ($sku) {
            $hasTax = false;
            $taxCode = $this->getTaxCodeItem($item_id);
            $xml = '<CreditMemoLineAdd>';
            $xml .= $this->multipleXml($sku, ['ItemRef', 'FullName'], 30);
            $xml .= $this->simpleXml($qty, 'Quantity');
            $xml .= $this->simpleXml($price, 'Rate');

            if ($taxAmount != 0) {
                $hasTax = true;
            }
            $xml .= $this->getXmlTax($taxCode, $hasTax);
            $xml .= '</CreditMemoLineAdd>';
        } else {
            $xml = '<CreditMemoLineAdd>';
            $xml .= $this->multipleXml('Not Found Product From M2: ' . $sku, ['ItemRef', 'ListID']);
            $xml .= '</CreditMemoLineAdd>';
            return $xml;
        }

        return $xml;
    }
}
